
const {response, request} = require('express');
const bcrypt = require('bcryptjs');
const { Op } = require("sequelize");

const {db} = require('../tables_db/init-models');

const { generarJWT } = require('../helpers/trigger-JWT');

const User = db.Users;
const Company = db.Company;

/* *************************************************************** */
/* **********************Services GET***************************** */

const getUsers = async (req = request, res = response ) => { 
  let {page, limit} = req.query;
  if(!page || !limit){
    page = 1;
    limit=10;
  }
  try {

  const users = await User.findAll({
    include:[{
      model: Company, 
      as:'companies',
      attributes: ['business_name'] 
    }],
    attributes:['email', 'name_person'],
    limit: parseInt(limit),
    offset: (page - 1) * limit
  });

  return res.json({
    ok: true,
    users
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      error
    });
  }
};

const getUserById = async (req = request, res = response ) => { 
  const {id} = req.params;
  try {

  const user = await User.findByPk(id);

  return res.json({
    ok: true,
    user
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};
const getUserJwt = async (req = request, res = response ) => { 
  const id = req.uid;
  try {

  const user = await User.findByPk(id);

  return res.json({
    ok: true,
    user
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services POST**************************** */

const postUser = async (req = request, res = response ) => { 

  const { body } = req;

  try {

    //Validar que no exista otra data igual
    const existData = await User.findOne({
      where :{
      email : body.email
    }
    });

    if(existData){
      return res.status(409).json({
        ok: false,
        msg:`El correo ya esta en uso.`
      });
    }

    const newUser = new User( body );

    //Encriptar contraseña
    let hashedPassword = bcrypt.genSaltSync();
    newUser.user_password = bcrypt.hashSync(body.user_password, hashedPassword);

    await newUser.save();

    const token = await generarJWT( newUser.id_user );

    return res.json({
        ok: true,
        newUser,
        token
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};
/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services PUT***************************** */

const putUser = async (req = request, res = response ) => { 

  const { id } = req.params.id;
  let {id_user, name_person, address_person, phone_person, email } = req.body;

  try {

    //Validar que exista la data para actualizar
    const existData = await User.findByPk( id_user );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe información para ser actualizada.`
      });
    }

    //Validar que no exista otro email igual
    
    if(email !== existData.email){
      const emailDB = await User.findOne({
        where :{
        email : body.email
      }
      });
      if( emailDB )
      return res.status(400).json({
        ok: false,
        msg:`El correo electrónico ya se encuentra en uso.`
      });
    }

    await existData.update({name_person, address_person, phone_person});

    return res.json({
      ok: true,
      msg: 'Información actualizada con éxito.',
      existData
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services DELETE************************** */

const deleteUser = async (req = request, res = response ) => { 

  const { id } = req.params.id;

  try {

    //Validar que exista la data para actualizar
    const existData = await User.findByPK( id );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe información para ser actualizada.`
      });
    }

    await existData.update({status_user: 0});

    return res.json({
      ok: true,
      msg: 'Usuario eliminado con éxito.'
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* ***************************EXPORTS***************************** */

module.exports = {
  getUsers,
  getUserById,
  getUserJwt,
  postUser,
  putUser,
  deleteUser
}
/* *************************************************************** */
/* *************************************************************** */