const {response, request} = require('express');

const { Op, Sequelize } = require("sequelize");

const {db} = require('../tables_db/init-models');

const Patient = db.Patient
const Company = db.Company;
const Attention = db.Attention;
const Exams = db.Exams;
const OrganSystemsReview = db.OrganSystemsReview;
const PathologicalBackground = db.PathologicalBackground;
const ReasonConsultation = db.ReasonConsultation;
const TreatmentPlan = db.TreatmentPlan;
const VitalSigns = db.VitalSigns;
const Diagnosis = db.Diagnosis;
const Evolution = db.Evolution;
const User = db.Users;

/* *************************************************************** */
/* **********************Services GET***************************** */

const getAttentionByPatient = async (req = request, res = response ) => { 
  const id_company = req.company;
  const id = req.params.id;
  let {page, limit} = req.query;
  if(!page || !limit){
    page = 1;
    limit=100;
  }
  try {

    const patients = await Patient.findAndCountAll({
      include: [
        {
          model: Company,
          as: 'companies',
          where: { id_company },
          through: {
            attributes: []
          }
        },
        {
          model: Attention,
          as: 'attention',
          order: [['create_time', 'DESC']]
        }
      ],
      attributes: ['names_patient', 'id_patient', 'document_number', 'phone'],
      where: {
        [Op.or]: [
          {
            names_patient: {
              [Op.like]: `%${id}%`
            }
          },
          {
            document_number: {
              [Op.like]: `%${id}%`
            }
          }
        ]
      },
      order: [['names_patient', 'ASC']],
      subQuery: false
    });
    const formattedPatients = patients.rows.map(patient => {
      return {
        names_patient: patient.names_patient,
        id_patient: patient.id_patient,
        document_number: patient.document_number,
        phone: patient.phone,
        attention: patient.attention.map(attention => ({
          id_attention: attention.id_attention,
          id_patient: attention.id_patient,
          id_company: attention.id_company,
          id_user: attention.id_user,
          observation: attention.observation,
          status_attention: attention.status_attention,
          create_time: attention.create_time
        }))
      };
    });

    // Ordenar el array por create_time en orden descendente
    formattedPatients.sort((a, b) => {
      const timeA = a.attention.length > 0 ? a.attention[0].create_time : 0;
      const timeB = b.attention.length > 0 ? b.attention[0].create_time : 0;
      return new Date(timeB) - new Date(timeA);
    });
    
    return res.json({
      ok: true,
      patients: {
        count: patients.count,
        rows: formattedPatients
      }
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getAttentionByPatientByDate = async (req = request, res = response ) => { 
  const id_company = req.company;
  const dateObject = req.params.start;
  const dateObject2 = req.params.end;
  let start = new Date(dateObject);
  let end = new Date(dateObject2); 

  let {page, limit} = req.query;
  page = parseInt(page) || 0;
  limit = parseInt(limit) || 10;

  try {

    const user = await User.findByPk(req.uid);

    const patients = await Patient.findAndCountAll({
      include: [
        {
          model: Company,
          as: 'companies',
          where: { id_company },
          through: {
            attributes: []
          },
          attributes: []
        },
        {
          model: Attention,
          as: 'attention',
          where: {
            [Op.and]:[
              {id_company},
              {create_time: {
                [Op.between]: [start, end]
              }}
            ]
          },
        }
      ],
      attributes: ['names_patient', 'id_patient', 'document_number', 'phone'],
      order: [['names_patient', 'ASC']],
      limit,
      offset: page * limit,
      subQuery: false
    });
    
    const formattedPatients = patients.rows.map(patient => {
      return {
        names_patient: patient.names_patient,
        id_patient: patient.id_patient,
        document_number: patient.document_number,
        phone: patient.phone,
        attention: patient.attention.map(attention => ({
          id_attention: attention.id_attention,
          id_patient: attention.id_patient,
          id_company: attention.id_company,
          id_user: attention.id_user,
          observation: attention.observation,
          status_attention: attention.status_attention,
          create_time: attention.create_time
        }))
      };
    });
    
    return res.json({
      ok: true,
      patients: {
        count: patients.count,
        rows: formattedPatients
      },
      user
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getAttention = async (req = request, res = response ) => { 
  const id_patient = req.params.id;
  const id_company = req.company;
  try {

  const attentions = await Attention.findAll({
    where:{
      [op.and]:[
        { id_patient },
        { id_company }

      ]
    },
    order: [
      ['create_time', 'DESC']
    ]
  });

  return res.json({
    ok: true,
    attentions
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getAttentionTotal = async (req = request, res = response ) => { 
  const id_patient = req.params.id;
  const id_company = req.company;
  try {

    const result = await Attention.findAll({
      attributes: ['id_attention', 'create_time'],
      include: [
        {
          model: Patient,
          as: 'patient',
          attributes: [],
          where: { id_patient }
        },
        {
          model: Company,
          as: 'companies',
          attributes: [],
          where: { id_company }
        },
      ],
      order: [['create_time', 'DESC']],
    });

    return res.json({
      ok: true,
      result
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getTotal = async (req = request, res = response ) => { 
  const id_attention = req.params.attention;
  try {

    const aux = await Attention.findByPk(id_attention);

    const id_patient = aux.dataValues.id_patient;

    const vital = await VitalSigns.findAll({
      attributes:[
        ['temperature', 'Temperatura'],
        [Sequelize.literal('CONCAT(bloodPressureA, "/", bloodPressureB)'), 'Presión Arterial'],
        ['pulse', 'Pulso'],
        ['breathingFrequency', 'Frecuencia Respiratoria'],
        ['weight', 'Peso'],
        ['imc', 'IMC'],
        ['abdominalPerimeter', 'Perímetro Abdominal'],
        ['hemoglobin', 'Hemoglobina'],
        ['glucose', 'Glucosa'],
        ['pulsioximetria', 'Pulsometria'],
      ],
      where:{
        id_attention
      }
    })

    // Consulta ReasonConsultation
    const reasonConsultationQuery = await ReasonConsultation.findAll({
      attributes: [
        ['reason', 'Motivo'],
        ['disease', 'Enfermedad Actual']
      ],
      where: { id_attention },
    });

    // Consulta TreatmentPlan
    const treatmentPlanQuery = await TreatmentPlan.findAll({
      attributes: [
        ['treatment_plan', 'Plan de tratamiento']
      ],
      where: { id_attention },
    });

    const evolution = await Evolution.findAll({
      attributes:[
        ['evolution', 'Evolución'],
        ['prescriptions', 'Prescripciones']
      ],
      where: { id_attention }
    });

    // Consulta Diagnosis
    const diagnosisQuery = await Diagnosis.findAll({
      attributes: [
        ['cie', 'CIE'],
        ['diagnosis', 'Diagnostico'],
        ['status_diagnosis', 'estado'],
      ],
      where: { id_attention },
      raw: true,
    });
  
    const transformedResult = diagnosisQuery.map(diagnosis => {
      const { CIE, Diagnostico, estado } = diagnosis;
      const Estado = estado === 0 ? 'Presuntivo' : 'Definitivo';
  
      return {
        CIE, 
        Diagnostico, 
        Estado
      };
    });

    // Construir la estructura JSON
    const result = {      
      Motivo: reasonConsultationQuery,
      'Plan De Tratamiento': treatmentPlanQuery,
      Diagnósticos: transformedResult,
      Evolution: evolution
    };

    return res.json({
      'Signos Vitales': vital,
      'Motivo De Consulta': reasonConsultationQuery,
      'Plan De Tratamiento': treatmentPlanQuery,
      Diagnósticos: transformedResult,
      Evolución: evolution
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getPatientByAttention = async (req = request, res = response ) => { 

  const id_company = req.company;
  const id_user = req.uid;
  try {
    let {page, limit} = req.query;
    page = parseInt(page) || 0;
    limit = parseInt(limit) || 10;

    const subquery = db.sequelize.literal(`(
      SELECT MAX(create_time) FROM Attention AS a
      WHERE a.id_patient = patient.id_patient
      AND a.id_company = ${id_company}
      AND a.id_user = ${id_user}
    )`);
    
    const attention = await Attention.findAll({
      attributes: [
        [subquery, 'maxDate'],
        [db.sequelize.col('patient.document_number'), 'document_number'],
        [db.sequelize.col('patient.names_patient'), 'names_patient'],
        [db.sequelize.col('patient.phone'), 'phone'],
      ],
      include: [
        {
          model: Patient,
          as: 'patient',
          attributes: [],
        },
      ],
      where: {
        id_company,
        id_user
      },
      group: ['patient.id_patient', 'patient.document_number', 'patient.names_patient', 'patient.phone'],
      order: [
        [db.sequelize.col('patient.names_patient'), 'ASC']
      ],
      offset: page * limit,
      limit,
      raw: true,
    });

    const count = await Attention.count({
      attributes: [
        [db.sequelize.fn('COUNT', db.sequelize.fn('DISTINCT', db.sequelize.col('id_patient'))), 'count']
      ],
      where: {
        id_company,
        id_user
      },
    });
    

    return res.json({
      ok: true,
      attention,
      count
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getLikePatientAttention = async( req= request, res = response ) => {
  const id = req.params.id;
  const id_company = req.company;
  const id_user = req.uid;
  try {
    
    const subquery = db.sequelize.literal(`(
      SELECT MAX(create_time) FROM Attention AS a
      WHERE a.id_patient = patient.id_patient
      AND a.id_company = ${id_company}
      AND a.id_user = ${id_user}
    )`);
    
    const attention = await Attention.findAll({
      attributes: [
        [subquery, 'maxDate'],
        [db.sequelize.col('patient.document_number'), 'document_number'],
        [db.sequelize.col('patient.names_patient'), 'names_patient'],
        [db.sequelize.col('patient.phone'), 'phone'],
      ],
      include: [
        {
          model: Patient,
          as: 'patient',
          attributes: [],
          where:{
            names_patient :{
              [Op.like]: `%${id}%`
            }
          }
        },
      ],
      where: {
        id_company,
        id_user
      },
      group: ['patient.id_patient', 'patient.document_number', 'patient.names_patient', 'patient.phone'],
      order: [
        [db.sequelize.col('patient.names_patient'), 'ASC']
      ],
      raw: true,
    });
    

    return res.json({
      ok: true,
      attention
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
}
/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services POST**************************** */


/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services PUT***************************** */


/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services DELETE************************** */


/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* ***************************EXPORTS***************************** */

module.exports = {
  getAttentionByPatient,
  getAttention,
  getAttentionByPatientByDate,
  getAttentionTotal,
  getTotal,
  getPatientByAttention,
  getLikePatientAttention
}
/* *************************************************************** */
/* *************************************************************** */