const {response, request} = require('express');

const { Op, Sequelize } = require("sequelize");

const {db, sequelize} = require('../tables_db/init-models');

const Medic = db.Medic;
const Medic_Company = db.Medic_Company;
const Company = db.Company;
const Address = db.Address;
const Phone = db.Phone;

/* *************************************************************** */
/* **********************Services GET***************************** */

const getMedic = async (req = request, res = response ) => { 
  const id_company = req.company;
  try {

    const medics = await Medic.findAll({
      attributes: [ [Sequelize.col('companies->Medic_Company.id_medic_company'), 'id_medic_company'],'id_medic', 'document_number', 'doctor_name'],
      include: [
        {
          model: Company,
          as: 'companies',
          through: {
            model: Medic_Company,
            where: { id_company },
            attributes: ['id_medic_company'],
          },
          attributes: [],
          required: true
        }
      ],
      order: [['doctor_name', 'ASC']]
    })

    return res.json({
      ok: true,
      medics
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getMedicCompanyByPk = async (req = request, res = response ) => {
  const document_number = req.params.id;
  try {

    const medic = await Medic.findOne({
      where:{
        document_number
      }
    });

    if( !medic ){
      return res.status(200).json({
        ok: false,
        msg: "No se encontró el médico"
      });
    }

    return res.json({
      ok: true,
      medic
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getMedicById = async (req = request, res = response ) => { 
  const id_medic = req.params.id;
  try {

  const medic = await Medic.findOne({
    where:{
      id_medic
    }
  });

  const address = await Address.findAll({
    where: {
      id_medic
    }
  })

  const phone = await Phone.findAll({
    where: {
      id_medic
    }
  })

  return res.json({
    ok: true,
    medic,
    address,
    phone
  });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services POST**************************** */

const postMedic = async (req = request, res = response ) => { 

  const { body } = req;
  const id_company = req.company;

  const medic = body.doctor;
  const address = body.address;
  const phone = body.phone;

  try {

    //Validar que no exista otra data igual
    const existData = await Medic.findOne({
      where :{
      document_number : medic.document_number
    }
    });

    if(existData){
      const data = await Medic_Company.findOne({
        where: {
          id_medic : existData.id_medic,
          id_company: existData.id_company
        }
      })
      if(!data){
        const medicCompany = new Medic_Company();
        medicCompany.id_company = id_company;
        medicCompany.id_medic = existData.id_medic;
        await medicCompany.save();
        return res.json({
          ok: true,
          msg: 'Médico Vinculado al Centro Médico con éxito.'
        });
      }else{
        return res.status(409).json({
          ok: false,
          msg: 'El médico ya se encuentra vinculado al Centro Médico.'
        });
      }
    }

    const newMedic = new Medic( medic );
    await newMedic.save();

    address.forEach(async i => {
      const newAddress = new Address();
      newAddress.id_medic = newMedic.id_medic;
      newAddress.id_type = i.id_type;
      newAddress.address_medic = i.address_medic;
      newAddress.reference = i.reference;
      await newAddress.save();
    });

    phone.forEach(async i => {
      const newPhone = new Phone();
      newPhone.id_medic = newMedic.id_medic;
      newPhone.id_type = i.id_type;
      newPhone.phone = i.phone;
      await newPhone.save();
    });

    const medicCompany = new Medic_Company();
    medicCompany.id_company = id_company;
    medicCompany.id_medic = newMedic.id_medic;
    await medicCompany.save();
    return res.json({
      ok: true,
      newMedic,
      msg: 'Médico creado y vinculado con éxito al Centro Médico.'
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services PUT***************************** */

const putMedic = async (req = request, res = response ) => { 

  const { body } = req;

  const medic = body.doctor;
  const address = body.address;
  const phone = body.phone;

  try {

    //Validar que exista la data para actualizar
    const existData = await Medic.findByPk( medic.id_medic );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe médico para ser actualizado.`
      });
    }

    await existData.update({id_specialty: medic.id_specialty, doctor_name: medic.doctor_name, email: medic.email, birthday_Doctor: medic.birthday_Doctor, id_civil_status: medic.id_civil_status});

    const addr = await Address.findAll({
      where:{
        id_medic: medic.id_medic
      }
    })

    await Promise.all(addr.map(async (address) => {
      await address.destroy();
  }));

    address.forEach(async i => {
      const newAddress = new Address();
      newAddress.id_medic = medic.id_medic;
      newAddress.id_type = i.id_type;
      newAddress.address_medic = i.address_medic;
      newAddress.reference = i.reference;
      await newAddress.save();
    });

    const pho = await Phone.findAll({
      where: {
        id_medic: medic.id_medic
      }
    })

    await Promise.all(pho.map(async (phone) => {
      await phone.destroy();
  }));

    phone.forEach(async i => {
      const newPhone = new Phone();
      newPhone.id_medic = medic.id_medic;
      newPhone.id_type = i.id_type;
      newPhone.phone = i.phone;
      await newPhone.save();
    });

    return res.json({
      ok: true,
      msg: 'Información del médico actualizada con éxito.',
      existData
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services DELETE************************** */

const deleteMedic = async (req = request, res = response ) => { 

  const id  = req.params.id;

  try {

    //Validar que exista la data para actualizar
    const existData = await Medic_Company.findByPk( id );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe médico para ser desvinculado.`
      });
    }

    await existData.destroy({id_medic_company: id});

    return res.json({
      ok: true,
      msg: 'Médico desvinculado con éxito.'
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* ***************************EXPORTS***************************** */

module.exports = {
  getMedic,
  getMedicCompanyByPk,
  postMedic,
  putMedic,
  deleteMedic,
  getMedicById
}
/* *************************************************************** */
/* *************************************************************** */