const {response, request} = require('express');

const { Op } = require("sequelize");

const {db} = require('../tables_db/init-models');

const Company = db.Company;



/* *************************************************************** */
/* **********************Services GET***************************** */

const getCompanies = async (req = request, res = response ) => { 
  let {page, limit} = req.query;
  try {
    
    if(!page || !limit){
      const companies = await Company.findAll({
        attributes:[ 'business_name', 'ruc'],
        where: {
          status_company: 1
        }
      });
      return res.json({
        ok: true, 
        companies
      });
    }else{
      const companies = await Company.findAndCountAll({
        attributes:[ 'business_name', 'ruc', 'logo' ],
        order: [
          ['business_name', 'ASC']
        ],
        limit: parseInt(limit),
        offset: (page - 1) * limit
      });
      return res.json({
        ok: true,
        companies
      });
    }


  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

const getCompanyByID = async (req = request, res = response ) => { 
  const {ruc} = req.params;
  try {

    const company = await Company.findOne({
      where:{ruc}
    });

    return res.json({
      ok: true,
      company
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services POST**************************** */

const postCompany = async (req = request, res = response ) => { 

  const { body } = req;

  try {

    //Validar que no exista otra data igual
    const existData = await Company.findOne({
      where :{
      ruc : body.ruc
    }
    });

    if(existData){
      return res.status(409).json({
        ok: false,
        msg:`El número de RUC: ${existData.ruc} ya esta registrado.`
      });
    }

    const newCompany = new Company( body );
    await newCompany.save();

    return res.json({
      ok: true,
      newCompany
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services PUT***************************** */

const putCompany = async (req = request, res = response ) => { 

  const { id } = req.params.id;
  let { business_name, trade_name, matrix_address, logo, obligatory_accounting, special_contributor } = req.body;

  try {

    //Validar que exista la data para actualizar
    const existData = await Company.findByPK( id );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`No existe información para ser actualizada.`
      });
    }

    await existData.update({business_name, trade_name, matrix_address, logo, obligatory_accounting, special_contributor});

    return res.json({
      ok: true,
      msg: 'Información de la empresa actualizada con éxito.',
      existData
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* **********************Services DELETE************************** */

const deleteCompany = async (req = request, res = response ) => { 

  const { id } = req.params.id;

  try {

    //Validar que exista la data para eliminar
    const existData = await Company.findByPK( id );

    if(!existData){
      return res.status(400).json({
        ok: false,
        msg:`La empresa que desea eliminar no existe o su identificador es incorrecto.`
      });
    }

    await existData.update({status_company: 0});

    return res.json({
      ok: true,
      msg: `La empresa ${existData.business_name}, fue eliminada con éxito.`
    });

  } catch (error) {
    console.log(error);
    return res.status(500).json({
      ok:false,
      msg: error
    });
  }
};

/* *************************************************************** */
/* *************************************************************** */


/* *************************************************************** */
/* ***************************EXPORTS***************************** */

module.exports = {
  getCompanies,
  getCompanyByID,
  postCompany,
  putCompany,
  deleteCompany
}
/* *************************************************************** */
/* *************************************************************** */